package main

import (
	"bytes"
	"crypto/rand"
	"crypto/sha256"
	"crypto/subtle"
	"encoding/hex"
	"errors"
	"fmt"
	"github.com/go-git/go-git/v5/utils/binary"
	"github.com/mholt/archiver/v3"
	"golang.org/x/term"
	"io"
	"net/http"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"syscall"
)

type outputwriter map[string]*bytes.Buffer

var outputmanager = make(outputwriter)

func (ow outputwriter) register(name string) *bytes.Buffer {
	ow[name] = bytes.NewBuffer(nil)
	return ow[name]
}

func (ow outputwriter) archive(name string) error {
	f, err := os.Create("out/" + name + ".log")
	if err != nil {
		return err
	}
	defer f.Close()
	ow[name].WriteTo(f)
	delete(ow, name)
	return nil
}

func (ow outputwriter) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "text/plain")
	name := r.URL.Path[len("/log/"):]
	if name == "" {
		http.Error(w, "No log name specified", 400)
		return
	}
	if outbuf, ok := ow[name]; ok {
		w.Write(outbuf.Bytes())
	} else {
		http.ServeFile(w, r, "out/"+name+".log")
	}
}

func uploadFile(w http.ResponseWriter, r *http.Request) {
	r.ParseMultipartForm(100 << 20)

	file, handler, err := r.FormFile("myFile")
	if err != nil {
		http.Error(w, err.Error(), 400)
		return
	}
	defer file.Close()

	folder := strings.TrimSuffix(handler.Filename, filepath.Ext(handler.Filename))
	if strings.HasSuffix(folder, ".tar") {
		folder = strings.TrimSuffix(folder, ".tar")
	}

	_, err = os.Stat(folder)
	if err == nil || folder == "" || folder == "." || folder == ".." || folder == "out" || folder == "malivveb" || folder == ".pass" {
		http.Error(w, "Project already exists", 400)
		return
	}

	dst, err := os.Create(handler.Filename)
	if err != nil {
		http.Error(w, err.Error(), 500)
		return
	}
	defer dst.Close()

	if _, err := io.Copy(dst, file); err != nil {
		http.Error(w, err.Error(), 500)
		return
	}

	go extractandbuild(handler.Filename)

	w.Header().Set("Content-Type", "text/html")
	w.Header().Set("charset", "utf-8")
	w.Header().Set("logfile", "/log/"+folder)
	w.WriteHeader(http.StatusOK)
	fmt.Fprintf(w, "<center><h3>Successfully Uploaded File</h3><br><a href=\"/log/%s\">View Log</a></center>", folder)
}

func extractandbuild(filename string) {

	folder := strings.TrimSuffix(filename, filepath.Ext(filename))
	if strings.HasSuffix(folder, ".tar") {
		folder = strings.TrimSuffix(folder, ".tar")
	}
	output := outputmanager.register(folder)

	err := archiver.Unarchive(filename, folder)
	if err != nil {
		output.WriteString("Could not extract archive: " + err.Error() + " Unable to build; exiting\n")
		return
	}
	defer os.RemoveAll(folder)
	output.WriteString("Archive extracted\n")

	err = os.Remove(filename)
	if err != nil {
		output.WriteString("Could not remove archive: " + err.Error() + "... Continuing anyway\n")
	}
	output.WriteString("Archive removed\n")

	err = os.Chmod(folder+"/build.sh", 0777)
	if err != nil {
		output.WriteString("build.sh could not be made exceutable: " + err.Error() + " Unable to build; exiting\n")
		return
	}
	output.WriteString("build.sh made executable\n")

	cmd := exec.Command("sh", "build.sh")
	cmd.Stdout = output
	cmd.Stderr = output
	cmd.Dir = folder
	err = cmd.Run()
	if err != nil {
		output.WriteString("build.sh failed to execute: " + err.Error() + " Unable to build; exiting\n")
		return
	}
	output.WriteString("build.sh executed\n")

	err = archiver.Archive([]string{folder}, "out/"+folder+".tar.gz")
	if err != nil {
		output.WriteString("Build files could not be archived :" + err.Error() + " Unable to make build files available; exiting\n")
		return
	}
	output.WriteString("Build files archived and moved into /out/" + folder + ".tar.gz\n")

	outputmanager.archive(folder)
}

func uploadHandler(w http.ResponseWriter, r *http.Request) {
	switch r.Method {
	case "GET":
		fmt.Fprintf(w, `<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <meta http-equiv="X-UA-Compatible" content="ie=edge" />
    <title>Upload File</title>
</head>
<body>
<form enctype="multipart/form-data" action="/upload" method="post">
    <input type="file" name="myFile" />
    <input type="submit" value="upload" />
</form>
</body>
</html>`)
	case "POST":
		uploadFile(w, r)
	}
}

func deleteHandler(w http.ResponseWriter, r *http.Request) {
	switch r.Method {
	case "GET":
		fmt.Fprintf(w, `<!DOCTYPE html>
<html lang="en">
<head>
	<title>Delete Project</title>
</head>
<body>
<form action="/delete" method="post">
	<input type="text" name="name" placeholder="Project Name">
	<input type="submit" value="Delete">
</form>
</body>
</html>`)
	case "POST":
		file := r.FormValue("name")
		os.Remove("out/" + file + ".tar.gz")
		os.Remove("out/" + file + ".log")
		fmt.Fprintf(w, "Deleted project %s", file)
	}
}

func readpass() (string, map[string]string) {
	pass, err := os.ReadFile(".pass")
	if err != nil {
		panic(err)
	}
	splitstr := strings.Split(string(pass), ":")
	if len(splitstr) != 3 {
		panic("Invalid .pass file")
	}
	userpass := make(map[string]string)
	userpass[splitstr[0]] = splitstr[1]
	return splitstr[2], userpass
}

func hashpass(password string, salt string) string {
	hasher := sha256.New()
	hasher.Write([]byte(password))
	checksum := hasher.Sum([]byte(salt))
	return hex.EncodeToString(checksum)
}

func createRandomSalt(n int) (string, error) {
	salt := make([]byte, n)
	if _, err := rand.Read(salt); err != nil {
		return "", errors.New("error generating salt")
	}
	return fmt.Sprintf("%x", salt), nil
}

func createpass() {
	_, err := os.Stat(".pass")
	if err == nil {
		return
	}
	salt, err := createRandomSalt(32)
	if err != nil {
		fmt.Println("Error generating salt")
		os.Exit(1)
	}
	fmt.Println("Enter username: ")
	b, _ := binary.ReadUntil(os.Stdin, '\n')
	username := string(b)
	fmt.Println("Enter password: ")
	b, _ = term.ReadPassword(int(syscall.Stdin))
	passwd1 := string(b)
	fmt.Println("Confirm password: ")
	b, _ = term.ReadPassword(int(syscall.Stdin))
	passwd2 := string(b)
	if passwd1 != passwd2 {
		fmt.Println("Passwords do not match")
		os.Exit(1)
	}
	passwd := hashpass(passwd1, salt)
	err = os.WriteFile(".pass", []byte(username+":"+passwd+":"+salt), 0600)
	if err != nil {
		fmt.Println("Error writing to .pass file")
		os.Exit(1)
	}
}

func main() {
	createpass()

	_, err := os.Stat("out")
	if err != nil {
		err = os.Mkdir("out", 0777)
		if err != nil {
			panic(err)
		}
	}

	port := "8080"
	if len(os.Args) > 1 && strings.HasPrefix(os.Args[1], "--port=") {
		port = strings.TrimPrefix(os.Args[1], "--port=")
		if port == "" {
			port = "8080"
		}
	}

	authenticateMe := func(next http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			salt, userpass := readpass()
			user, apass, authOK := r.BasicAuth()
			hasher := sha256.New()
			hasher.Write([]byte(apass))
			checksum := hasher.Sum([]byte(salt))
			pass := hex.EncodeToString(checksum)
			expectedPass, lookupOK := userpass[user]

			if !authOK || !lookupOK || subtle.ConstantTimeCompare([]byte(expectedPass), []byte(pass)) != 1 {
				w.Header().Set("WWW-Authenticate", `Basic realm="Restricted"`)
				http.Error(w, "Unauthorized.", http.StatusUnauthorized)
				return
			}
			next.ServeHTTP(w, r)
		})
	}

	http.Handle("/log/", authenticateMe(outputmanager))
	http.Handle("/upload", authenticateMe(http.HandlerFunc(uploadHandler)))
	http.Handle("/out/", authenticateMe(http.StripPrefix("/out/", http.FileServer(http.Dir("out")))))
	http.Handle("/delete", authenticateMe(http.HandlerFunc(deleteHandler)))
	println("Listening on port " + port)
	http.ListenAndServe("0.0.0.0:"+port, nil)
}